import sys
import subprocess
import time
import os
import shlex
NEEDED_PACKAGES_MAP = {
    "PyQt5": "PyQt5",
    "PyQtWebEngine": "PyQtWebEngine",
    "tzdata": "tzdata",
    "pyautogui": "pyautogui",
    "pyperclip": "pyperclip",
    "pygetwindow": "PyGetWindow",
    "websockets": "websockets",
    "discord": "discord.py",
    "openai": "openai",
    "gpt4all": "gpt4all",
    "requests": "requests",
    "bs4": "beautifulsoup4",
    "psutil": "psutil",
}

NON_PIP_INSTALLABLE = {
    "tkinter",
}

IMPORT_CHECK_LIST = [
    "os",
    "sys",
    "shutil",
    "subprocess",
    "threading",
    "tempfile",
    "zipfile",
    "time",
    "ctypes",
    "shlex",
    "pathlib",
    "urllib.parse",
    "tkinter",
    "tkinter.ttk",
    "tkinter.messagebox",
    "tkinter.scrolledtext",
    "tkinter.filedialog",
]


def run_subprocess_stream(cmd, on_stdout_line=None):
    proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1, universal_newlines=True)
    for line in proc.stdout:
        if on_stdout_line:
            on_stdout_line(line.rstrip("\n"))
        else:
            print(line.rstrip("\n"))
    proc.wait()
    return proc.returncode


def pip_install_package(pkg_name):
    print(f"[installer] pip install {pkg_name} --user ...")
    if getattr(sys, 'frozen', False):
        print("[installer] frozen executable: 自動 pip インストールはサポートされていません。")
        return False
    cmd = [sys.executable, "-m", "pip", "install", "--upgrade", "--user", pkg_name]
    rc = run_subprocess_stream(cmd)
    return rc == 0


def import_module_by_name(name):
    import importlib
    return importlib.import_module(name)


def safe_top_module(name):
    return name.split(".")[0]


def ensure_imports(import_list):
    failed = []
    frozen = getattr(sys, 'frozen', False)
    for name in import_list:
        try:
            import_module_by_name(name)
        except Exception as e:
            top = safe_top_module(name)
            print(f"[installer] import {name!r} に失敗: {e!r}")
            if top in NON_PIP_INSTALLABLE:
                print(f"[installer] モジュール {top!r} はシステム依存のため自動インストールできません。")
                failed.append((name, None))
                continue
            pip_name = NEEDED_PACKAGES_MAP.get(top, top)
            if frozen:
                print(f"[installer] frozen 実行: {pip_name!r} の自動インストールはできません。")
                failed.append((name, pip_name))
                continue
            print(f"[installer] attempt pip install {pip_name!r} ... (module: {name})")
            ok = pip_install_package(pip_name)
            if not ok:
                print(f"[installer] pip install {pip_name!r} に失敗しました。")
                failed.append((name, pip_name))
            else:
                try:
                    import_module_by_name(name)
                except Exception as e2:
                    print(f"[installer] pip install 後も import に失敗: {e2!r}")
                    failed.append((name, pip_name))
    if failed:
        msg_lines = ["以下のモジュールの import/install に失敗しました:"]
        for mod, pkg in failed:
            if pkg:
                msg_lines.append(f"  module={mod}  pip_package={pkg}")
            else:
                msg_lines.append(f"  module={mod}  pip_package=None")
        raise RuntimeError("\n".join(msg_lines))
    return True

try:
    print("[installer] 必要モジュールのチェックを開始します...")
    ensure_imports(IMPORT_CHECK_LIST)
    print("[installer] 必要モジュールがそろいました。続行します。")
except Exception as e:
    print("[installer] 重要: 必要モジュールの準備に失敗しました。")
    print(repr(e))
    print("インストールを中止します。")
    sys.exit(1)

import shutil
import threading
import tempfile
import zipfile
import ctypes
from pathlib import Path
from urllib.parse import urlparse

import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext, filedialog

DA_UI_FOLDER = r"C:\Program Files\DA_UI"
REMOTE_VERSION_URL = "https://ud-autumn.site/DA/app/version.txt"
REMOTE_ZIP_URL = "https://ud-autumn.site/DA/app/DA_UI.zip"
LICENSE_FILENAME = "利用規約.txt"
SHORTCUT_NAME = "DA_UI.lnk"
NEEDED_PACKAGES_MAP.update({})
MIN_PYVER = (3, 10)
ELEVATED_FLAG = '--elevated-run'
ELEVATED_ACTION_FLAG = '--elevated-action'


def is_windows():
    return os.name == 'nt'


def is_admin():
    if not is_windows():
        return False
    try:
        return ctypes.windll.shell32.IsUserAnAdmin() != 0
    except Exception:
        return False


def elevate_via_shell_execute(argv=None, action=None):
    if not is_windows():
        return False
    if argv is None:
        argv = sys.argv
    is_frozen = getattr(sys, 'frozen', False)
    if is_frozen:
        executable = os.path.abspath(argv[0])
        rest_args = list(argv[1:])
        if ELEVATED_FLAG not in rest_args:
            rest_args.append(ELEVATED_FLAG)
        if action:
            rest_args.extend([ELEVATED_ACTION_FLAG, action])
        params = subprocess.list2cmdline(rest_args)
    else:
        executable = sys.executable
        script = os.path.abspath(argv[0])
        rest = argv[1:]
        if ELEVATED_FLAG not in rest:
            rest = rest + [ELEVATED_FLAG]
        if action:
            rest = rest + [ELEVATED_ACTION_FLAG, action]
        params = f'"{script}" ' + (subprocess.list2cmdline(rest) if rest else "")
    try:
        rc = ctypes.windll.shell32.ShellExecuteW(None, "runas", executable, params, None, 1)
        try:
            rc_int = int(rc)
        except Exception:
            rc_int = 0
        if rc_int > 32:
            os._exit(0)
        else:
            return False
    except Exception:
        return False
    

def get_elevated_action_from_argv(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        idx = argv.index(ELEVATED_ACTION_FLAG)
        if idx + 1 < len(argv):
            return argv[idx + 1]
    except ValueError:
        pass
    return None


def run_subprocess_stream_ui(cmd, on_stdout_line=None):
    proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1, universal_newlines=True)
    for line in proc.stdout:
        if on_stdout_line:
            on_stdout_line(line.rstrip("\n"))
    proc.wait()
    return proc.returncode


def find_installer_dir():
    if getattr(sys, '_MEIPASS', None):
        return os.path.abspath(sys._MEIPASS)
    if getattr(sys, 'frozen', False):
        return os.path.dirname(os.path.abspath(sys.executable))
    return os.path.dirname(os.path.abspath(sys.argv[0]))


def vbs_create_shortcut(lnk_path, target, args="", working_dir=None, icon=None):
    def esc(s):
        return s.replace('"', '""') if s is not None else ""
    lnk_esc = esc(lnk_path)
    target_esc = esc(target)
    args_esc = esc(args)
    workdir_line = f'link.WorkingDirectory = "{esc(working_dir)}"' if working_dir else ""
    icon_line = f'link.IconLocation = "{esc(icon)}"' if icon else ""
    vbs = (
        'Set objShell = WScript.CreateObject("WScript.Shell")\n'
        f'Set link = objShell.CreateShortcut("{lnk_esc}")\n'
        f'link.TargetPath = "{target_esc}"\n'
        f'link.Arguments = "{args_esc}"\n'
        f'{workdir_line}\n'
        f'{icon_line}\n'
        'link.Save\n'
    )
    fd, tmpvbs = tempfile.mkstemp(suffix=".vbs", text=True)
    os.close(fd)
    try:
        with open(tmpvbs, "w", encoding="mbcs", errors="replace") as f:
            f.write(vbs)
        subprocess.run(["cscript", "//Nologo", tmpvbs], check=True)
        return True
    except Exception:
        try:
            workdir_ps = f'$sc.WorkingDirectory = "{working_dir}"' if working_dir else ''
            icon_ps = f'$sc.IconLocation = "{icon}"' if icon else ''
            ps = (
                '$W = New-Object -ComObject WScript.Shell\n'
                f'$sc = $W.CreateShortcut("{lnk_path}")\n'
                f'$sc.TargetPath = "{target}"\n'
                f'$sc.Arguments = "{args}"\n'
                f'{workdir_ps}\n'
                f'{icon_ps}\n'
                '$sc.Save()\n'
            )
            subprocess.run(["powershell", "-NoProfile", "-ExecutionPolicy", "Bypass", "-Command", ps], check=True)
            return True
        except Exception:
            return False
    finally:
        try:
            os.remove(tmpvbs)
        except:
            pass

def is_store_python_path(p):
    if not p:
        return False
    p = p.lower()
    return "windowsapps" in p and "pythonsoftwarefoundation" in p

def find_better_python(preferred=None):
    candidates = []
    if preferred:
        candidates.append(preferred)
    common = [
        r"C:\Program Files\Python311\pythonw.exe",
        r"C:\Program Files\Python311\python.exe",
        r"C:\Program Files\Python310\pythonw.exe",
        r"C:\Program Files\Python310\python.exe",
    ]
    candidates.extend(common)
    for name in ("pythonw", "python"):
        resolved = shutil.which(name)
        if resolved:
            candidates.append(resolved)
    candidates.append(sys.executable)
    for c in candidates:
        if c and os.path.exists(c):
            if is_store_python_path(c):
                continue
            return c
    return sys.executable

def create_shortcut_to_ui(shortcut_folder, python_exe=None):
    icon_path = os.path.join(DA_UI_FOLDER, "da_icon.ico")
    if getattr(sys, 'frozen', False):
        target_exec = os.path.abspath(sys.executable)
        args = ""
    else:
        # 変更点: より良い python を探す
        chosen_python = find_better_python(preferred=python_exe)
        exe_dir = os.path.dirname(chosen_python)
        pythonw = os.path.join(exe_dir, "pythonw.exe")
        if os.path.exists(pythonw):
            target_exec = pythonw
        else:
            target_exec = chosen_python
        ui_py = os.path.join(DA_UI_FOLDER, "ui.py")
        ui_exe = os.path.join(DA_UI_FOLDER, "ui.exe")
        if os.path.exists(ui_exe):
            args = ""
            target_exec = ui_exe
        else:
            if not os.path.exists(ui_py):
                return False, f"ui.py が見つかりません: {ui_py}"
            args = f'"{ui_py}"'
    if os.path.exists(icon_path):
        icon_to_use = icon_path
    else:
        icon_to_use = target_exec
    lnk_path = os.path.join(shortcut_folder, SHORTCUT_NAME)
    ok = vbs_create_shortcut(lnk_path, target_exec, args=args, working_dir=DA_UI_FOLDER, icon=icon_to_use)
    if ok:
        return True, lnk_path
    else:
        return False, "ショートカット作成に失敗しました。"


def download_with_progress(url, dest_path, on_progress=None):
    import requests
    with requests.get(url, stream=True, timeout=30) as r:
        r.raise_for_status()
        total = r.headers.get("Content-Length")
        total = int(total) if total and total.isdigit() else None
        received = 0
        tmp = dest_path + ".part"
        with open(tmp, "wb") as f:
            for chunk in r.iter_content(chunk_size=8192):
                if not chunk:
                    continue
                f.write(chunk)
                received += len(chunk)
                if on_progress:
                    on_progress(received, total)
        os.replace(tmp, dest_path)
    return dest_path


def ensure_dir_writable(path):
    try:
        testfile = os.path.join(path, f".perm_test_{int(time.time())}")
        with open(testfile, "w") as f:
            f.write("test")
        os.remove(testfile)
        return True
    except Exception:
        return False

class InstallerGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("DA_UI Installer")
        self.root.geometry("760x520")
        self.frame = ttk.Frame(root, padding=8)
        self.frame.pack(fill="both", expand=True)
        self.status_label = ttk.Label(self.frame, text="準備中...")
        self.status_label.pack(fill="x")
        self.log = scrolledtext.ScrolledText(self.frame, height=18)
        self.log.pack(fill="both", expand=True, pady=6)
        self.progress = ttk.Progressbar(self.frame, mode="determinate")
        self.progress.pack(fill="x", pady=4)
        self.button_frame = ttk.Frame(self.frame)
        self.button_frame.pack(fill="x", pady=6)
        self.action_button = ttk.Button(self.button_frame, text="開始", command=self.on_start)
        self.action_button.pack(side="left")
        self.cancel_button = ttk.Button(self.button_frame, text="終了", command=self.on_quit)
        self.cancel_button.pack(side="right")
        self.shortcut_folder = os.path.dirname(find_installer_dir())
        os.makedirs(self.shortcut_folder, exist_ok=True)
        self._stop_requested = False
        self._thread = None
        self.log_write("インストーラを起動しました。チェックを開始します...")
        self.root.after(100, self.initial_check_async)

    def log_write(self, s):
        ts = time.strftime("%H:%M:%S")
        self.log.insert("end", f"[{ts}] {s}\n")
        self.log.yview_moveto(1.0)

    def set_status(self, s):
        self.status_label.config(text=s)
        self.log_write(s)

    def set_progress(self, fraction, text=None):
        if fraction is None:
            self.progress.config(mode="indeterminate")
            try:
                self.progress.start(10)
            except Exception:
                pass
        else:
            self.progress.config(mode="determinate")
            self.progress.stop()
            self.progress['maximum'] = 1000
            self.progress['value'] = int(max(0, min(1.0, fraction)) * 1000)
        if text:
            self.set_status(text)

    def on_quit(self):
        if messagebox.askokcancel("終了", "インストーラを終了しますか？"):
            self._stop_requested = True
            try:
                self.root.quit()
            except:
                os._exit(0)

    def initial_check_async(self):
        def job():
            try:
                self.initial_check()
            except Exception as e:
                self.log_write("チェック中にエラー: " + repr(e))
        t = threading.Thread(target=job, daemon=True)
        t.start()

    def initial_check(self):
        self.set_status("既存インストールを確認しています...")
        installed = os.path.isdir(DA_UI_FOLDER)
        local_version = None
        if installed:
            local_version_path = os.path.join(DA_UI_FOLDER, "version.txt")
            if os.path.exists(local_version_path):
                try:
                    with open(local_version_path, "r", encoding="utf-8") as f:
                        local_version = f.read().strip()
                except:
                    local_version = None
        self.log_write("remote の version.txt を取得します。requests が無ければインストールします。")
        if not self.ensure_packages_sync(['requests']):
            self.log_write("requests のインストールに失敗しました。ネットワーク接続や権限を確認してください。")
            return
        import requests
        remote_version = None
        try:
            r = requests.get(REMOTE_VERSION_URL, timeout=15)
            if r.status_code == 200:
                remote_version = r.text.strip()
                self.log_write(f"remote version: {remote_version!r}")
            else:
                self.log_write(f"version.txt の取得に失敗しました: HTTP {r.status_code}")
        except Exception as e:
            self.log_write("version.txt 取得エラー: " + repr(e))
        if ELEVATED_FLAG in sys.argv:
            action = get_elevated_action_from_argv()
            if action == 'uninstall':
                self.set_status("管理者権限で再起動されました。アンインストールを続行します。")
                self._thread = threading.Thread(target=self.perform_uninstall, daemon=True)
                self._thread.start()
                return
            elif action == 'install':
                self.set_status("管理者権限で再起動されました。インストールを続行します。")
                self._thread = threading.Thread(target=self.install_flow, daemon=True)
                self._thread.start()
                return
            else:
                self.log_write(f"unknown elevated action: {action!r}; 通常フローへ戻ります。")
        if installed and local_version is not None and remote_version is not None and local_version == remote_version:
            self.set_status("同じバージョンがインストールされています。")
            self.show_installed_options()
            return
        else:
            self.set_status("新規インストール / アップデートを行います。利用規約を表示します。")
            self.show_license_and_consent()

    def show_installed_options(self):
        def on_do_nothing():
            dlg.destroy()
            self.set_status("何もしませんでした。")
        def on_recreate_shortcut():
            dlg.destroy()
            self.set_status("ショートカットを作成します...")
            ok, info = create_shortcut_to_ui(self.shortcut_folder)
            if ok:
                messagebox.showinfo("完了", f"ショートカットを作成しました: {info}")
                self.set_status("ショートカットを作成しました。")
            else:
                messagebox.showerror("エラー", f"ショートカット作成に失敗しました: {info}")
                self.set_status("ショートカット作成失敗。")
        def on_uninstall():
            dlg.destroy()
            if messagebox.askyesno("アンインストール", f"{DA_UI_FOLDER} を完全に削除します。よろしいですか？"):
                self.set_status("アンインストール中...")
                try:
                    if not is_admin():
                        self.log_write("アンインストールに管理者権限が必要です。昇格します...")
                        elevate_via_shell_execute(action='uninstall')
                        return
                    self.perform_uninstall()
                except Exception as e:
                    messagebox.showerror("エラー", "アンインストールに失敗しました。\n" + repr(e))
                    self.log_write("アンインストールエラー: " + repr(e))
        dlg = tk.Toplevel(self.root)
        dlg.title("インストール済み")
        ttk.Label(dlg, text="DA_UI は既に同バージョンがインストールされています。").pack(padx=12, pady=8)
        btnf = ttk.Frame(dlg)
        btnf.pack(padx=12, pady=12)
        ttk.Button(btnf, text="何もしない", command=on_do_nothing).pack(side="left", padx=6)
        ttk.Button(btnf, text="ショートカットを作り直す", command=on_recreate_shortcut).pack(side="left", padx=6)
        ttk.Button(btnf, text="アンインストール", command=on_uninstall).pack(side="left", padx=6)
        ttk.Button(btnf, text="閉じる", command=on_do_nothing).pack(side="left", padx=6)

    def show_license_and_consent(self):
        url = "https://ud-autumn.site/DA/document/TOU_PP.html"
        dlg = tk.Toplevel(self.root)
        dlg.title("利用規約")
        dlg.geometry("520x220")
        msg = "利用規約およびプライバシーポリシーに同意した場合、下記の同意ボタンを押してください。"
        ttk.Label(dlg, text=msg, wraplength=480, justify="left").pack(padx=12, pady=(12, 6))
        link_label = tk.Label(dlg, text="利用規約・プライバシーポリシーを表示する", fg="blue", cursor="hand2", underline=True)
        link_label.pack(padx=12, pady=(0, 8))
        def open_url(event=None):
            import webbrowser
            webbrowser.open(url)
        link_label.bind("<Button-1>", open_url)
        btnf = ttk.Frame(dlg)
        btnf.pack(fill="x", pady=12, padx=12)
        def on_cancel():
            dlg.destroy()
            self.set_status("利用規約に同意しませんでした。終了します。")
        def on_agree():
            dlg.destroy()
            self.set_status("利用規約に同意しました。インストールを続行します。")
            self._thread = threading.Thread(target=self.install_flow, daemon=True)
            self._thread.start()
        ttk.Button(btnf, text="同意してインストール", command=on_agree).pack(side="right", padx=6)
        ttk.Button(btnf, text="同意しない（終了）", command=on_cancel).pack(side="right", padx=6)


    def ensure_packages_sync(self, pkg_names):
        to_install = []
        frozen = getattr(sys, 'frozen', False)
        for name in pkg_names:
            pip_name = NEEDED_PACKAGES_MAP.get(name, name)
            import importlib.util
            modname = name if name in NEEDED_PACKAGES_MAP.keys() else pip_name.split()[0]
            spec = importlib.util.find_spec(modname)
            if spec is None:
                to_install.append(pip_name)
        if not to_install:
            self.log_write("必要なパッケージは既にインストールされています。")
            return True
        self.log_write("不足パッケージをインストールします: " + ", ".join(to_install))
        for pkg in to_install:
            if frozen:
                self.log_write(f"実行ファイル版: {pkg} を自動インストールできません。インストーラ(.py)で実行するか依存を同梱してください。")
                return False
            self.set_status(f"pip install {pkg} を実行中...")
            self.set_progress(None, f"{pkg} をインストール中...")
            cmd = [sys.executable, "-m", "pip", "install", "--user", pkg]
            def on_line(line):
                self.log_write(line)
            rc = run_subprocess_stream_ui(cmd, on_stdout_line=on_line)
            self.set_progress(0)
            if rc != 0:
                self.log_write(f"pip install {pkg} が失敗しました（戻り値 {rc}）。")
                return False
        return True

    def install_flow(self):
        if sys.version_info < MIN_PYVER:
            self.set_status(f"Python {MIN_PYVER[0]}.{MIN_PYVER[1]} 以上が必要です。現在: {sys.version_info.major}.{sys.version_info.minor}")
            if messagebox.askyesno("Python のバージョン", "このコンピュータの Python バージョンが古いです。ダウンロードページを開きますか？"):
                import webbrowser
                webbrowser.open("https://www.python.org/downloads/windows/")
            return

        self.set_status("必要パッケージの確認・インストールを行います...")
        keys = list(NEEDED_PACKAGES_MAP.keys())
        if not self.ensure_packages_sync(keys):
            messagebox.showerror("エラー", "必要なパッケージのインストールに失敗しました。ログを確認してください。")
            return

        self.set_status("DA_UI の ZIP をダウンロードします...")
        self.set_progress(0)
        try:
            tmpdir = tempfile.mkdtemp(prefix="da_ui_inst_")
            zip_dest = os.path.join(tmpdir, "DA_UI.zip")
            def on_prog(recv, total):
                if total:
                    frac = recv / total
                    self.set_progress(frac, f"ダウンロード中: {int(frac*100)}%")
                else:
                    self.set_progress(None, "ダウンロード中...")
            download_with_progress(REMOTE_ZIP_URL, zip_dest, on_progress=on_prog)
            self.set_progress(1.0, "ダウンロード完了")
            self.log_write("ダウンロード完了: " + zip_dest)
        except Exception as e:
            self.log_write("ダウンロードに失敗しました: " + repr(e))
            messagebox.showerror("エラー", "ZIP のダウンロードに失敗しました。\n" + repr(e))
            return

        self.set_status("解凍と配置を行います（管理者権限が必要な場合があります）...")
        try:
            if os.path.exists(DA_UI_FOLDER):
                try:
                    if is_admin():
                        shutil.rmtree(DA_UI_FOLDER)
                    else:
                        self.log_write("配置に管理者権限が必要です。昇格します...")
                        elevate_via_shell_execute(action='install')
                        return
                except Exception as e:
                    self.log_write("既存フォルダ削除エラー: " + repr(e))
                    messagebox.showerror("エラー", "既存フォルダの削除に失敗しました。\n" + repr(e))
                    return
            extract_tmp = os.path.join(tmpdir, "extracted")
            os.makedirs(extract_tmp, exist_ok=True)
            with zipfile.ZipFile(zip_dest, "r") as z:
                members = z.namelist()
                total = len(members)
                for idx, member in enumerate(members, start=1):
                    z.extract(member, path=extract_tmp)
                    if idx % 20 == 0 or idx == total:
                        self.set_progress(idx/total, f"解凍中: {int(idx/total*100)}%")
            if not is_admin():
                self.log_write("Program Files への配置には管理者権限が必要です。昇格します...")
                elevate_via_shell_execute(action='install')
                return
            maybe_top = os.path.join(extract_tmp, "DA_UI")
            if os.path.isdir(maybe_top):
                src_root = maybe_top
            else:
                entries = [e for e in os.listdir(extract_tmp) if e not in (".", "..")] 
                if len(entries) == 1 and os.path.isdir(os.path.join(extract_tmp, entries[0])):
                    src_root = os.path.join(extract_tmp, entries[0])
                else:
                    src_root = extract_tmp
            if os.path.exists(DA_UI_FOLDER):
                shutil.rmtree(DA_UI_FOLDER)
            shutil.move(src_root, DA_UI_FOLDER)
            self.set_progress(1.0, "配置完了")
            self.log_write(f"配置しました: {DA_UI_FOLDER}")
        except Exception as e:
            self.log_write("解凍/配置中にエラー: " + repr(e))
            messagebox.showerror("エラー", "解凍/配置に失敗しました。\n" + repr(e))
            return
        finally:
            try:
                shutil.rmtree(tmpdir)
            except:
                pass

        self.set_status("データ② を実行します（ファイアウォール設定など）...")
        try:
            data2_source = DATA2_CODE
            tf = tempfile.NamedTemporaryFile("w", delete=False, suffix=".py", encoding="utf-8")
            tf.write(data2_source)
            tf.flush()
            tf.close()
            if not is_admin():
                self.log_write("データ② の実行に管理者権限が必要です。昇格します...")
                elevate_via_shell_execute([sys.executable, tf.name])
                return
            rc = run_subprocess_stream_ui([sys.executable, tf.name, "--list"], on_stdout_line=lambda l: self.log_write(l))
            self.log_write(f"データ② 実行終了 code={rc}")
            try:
                os.remove(tf.name)
            except:
                pass
        except Exception as e:
            self.log_write("データ② 実行エラー: " + repr(e))

        self.set_status("ショートカットを作成します...")
        ok, info = create_shortcut_to_ui(self.shortcut_folder)
        if ok:
            messagebox.showinfo("インストール完了", "インストールが完了しました。\nショートカット: " + info)
            self.set_status("インストール完了。")
        else:
            messagebox.showwarning("注意", "インストールは完了しましたが、ショートカットの作成に失敗しました。\n" + str(info))
            self.set_status("インストール完了（ショートカット失敗）。")

    def perform_uninstall(self):
        try:
            if os.path.exists(DA_UI_FOLDER):
                shutil.rmtree(DA_UI_FOLDER)
                try:
                    messagebox.showinfo("完了", "アンインストールが完了しました。")
                except:
                    pass
                self.set_status("アンインストール完了。")
            else:
                try:
                    messagebox.showinfo("情報", "フォルダが見つかりませんでした。")
                except:
                    pass
                self.set_status("フォルダが見つかりませんでした。")
        except Exception as e:
            try:
                messagebox.showerror("エラー", "アンインストールに失敗しました。\n" + repr(e))
            except:
                pass
            self.log_write("アンインストールエラー: " + repr(e))

    def on_start(self):
        self.show_license_and_consent()

DATA2_CODE = r'''
import sys
import ctypes
import subprocess
import shlex
DEFAULT_PORT = 19132
DEFAULT_PROTOCOL = "UDP"
DEFAULT_NAME_BASE = "Minecraft Bedrock"

def is_windows():
    return sys.platform == "win32"

def is_admin():
    try:
        return ctypes.windll.shell32.IsUserAnAdmin() != 0
    except Exception:
        return False

def elevate_to_admin():
    params = " ".join(shlex.quote(arg) for arg in sys.argv)
    ctypes.windll.shell32.ShellExecuteW(None, "runas", sys.executable, params, None, 1)
    sys.exit(0)

def run_powershell(cmd):
    proc = subprocess.run(["powershell", "-NoProfile", "-ExecutionPolicy", "Bypass", "-Command", cmd], capture_output=True, text=True)
    return proc

def run_netsh(args_list):
    proc = subprocess.run(["netsh"] + args_list, capture_output=True, text=True)
    return proc

def add_rule(port=DEFAULT_PORT, protocol=DEFAULT_PROTOCOL, name=None):
    if name is None:
        name = f"{DEFAULT_NAME_BASE} ({port}/{protocol})"
    ps_cmd = f'New-NetFirewallRule -DisplayName "{name}" -Direction Inbound -Protocol {protocol} -LocalPort {port} -Action Allow -Profile Any'
    r = run_powershell(ps_cmd)
    if r.returncode == 0:
        print(f'[OK] PowerShell: ルールを追加しました: "{name}" (ポート {port} / {protocol})')
        return True
    else:
        print("[WARN] PowerShell による追加が失敗しました。stderr:\n", r.stderr.strip())
        ns_args = ["advfirewall", "firewall", "add", "rule", f'name={name}', "dir=in", "action=allow", f"protocol={protocol}", f"localport={port}"]
        r2 = run_netsh(ns_args)
        if r2.returncode == 0:
            print(f'[OK] netsh: ルールを追加しました: "{name}" (ポート {port} / {protocol})')
            return True
        else:
            print("[ERROR] netsh による追加も失敗しました。stderr:\n", r2.stderr.strip())
            return False

def remove_rule(name_pattern=DEFAULT_NAME_BASE):
    ps_cmd = f'Get-NetFirewallRule -DisplayName "*{name_pattern}*" | Remove-NetFirewallRule -ErrorAction SilentlyContinue'
    r = run_powershell(ps_cmd)
    if r.returncode == 0:
        print(f'[OK] PowerShell: 名前に "{name_pattern}" を含むルールを削除しました。')
        return True
    else:
        print("[WARN] PowerShell による削除が失敗しました。stderr:\n", r.stderr.strip())
        ns_args = ["advfirewall", "firewall", "delete", "rule", f"name={name_pattern}"]
        r2 = run_netsh(ns_args)
        if r2.returncode == 0:
            print(f'[OK] netsh: ルールを削除しました (name={name_pattern}).')
            return True
        else:
            print("[ERROR] netsh による削除も失敗しました。stderr:\n", r2.stderr.strip())
            return False

def list_rules(name_pattern=DEFAULT_NAME_BASE):
    ps_cmd = f'Get-NetFirewallRule -DisplayName "*{name_pattern}*" | Format-Table -AutoSize | Out-String -Width 4096'
    r = run_powershell(ps_cmd)
    if r.returncode == 0 and r.stdout.strip():
        print(r.stdout)
    else:
        print("[WARN] PowerShell での一覧表示が失敗または結果なしです。stderr:\n", r.stderr.strip())
        ns_args = ["advfirewall", "firewall", "show", "rule", f"name={name_pattern}"]
        r2 = run_netsh(ns_args)
        if r2.stdout.strip():
            print(r2.stdout)
        else:
            print("[INFO] netsh の出力:\n", r2.stderr.strip())

def apply_extra_powershell_settings():
    cmds = [
        'CheckNetIsolation LoopbackExempt -a -n="Microsoft.MinecraftUWP_8wekyb3d8bbwe"',
        'New-NetFirewallRule -DisplayName "Allow Minecraft Bedrock (UDP 19132)" -Direction Inbound -Protocol UDP -LocalPort 19132 -Action Allow -Profile Any',
        'New-NetFirewallRule -DisplayName "Allow_WS_6752" -Direction Inbound -Protocol TCP -LocalPort 6752 -Action Allow -Profile Any',
        'Get-NetFirewallRule -DisplayName "Allow_WS_6752" | Format-List'
    ]
    for c in cmds:
        r = run_powershell(c)
        if r.returncode == 0:
            if r.stdout and r.stdout.strip():
                print(r.stdout)
            else:
                print(f'[OK] PowerShell executed: {c}')
        else:
            print('[WARN] PowerShell failed for command:', c)
            if r.stderr and r.stderr.strip():
                print(r.stderr)

def parse_args_manual(argv):
    opts = {'action': None, 'port': DEFAULT_PORT, 'protocol': DEFAULT_PROTOCOL, 'name': None}
    i = 1
    while i < len(argv):
        a = argv[i]
        if a == '--add':
            opts['action'] = 'add'
            i += 1
        elif a == '--remove':
            opts['action'] = 'remove'
            i += 1
        elif a == '--list':
            opts['action'] = 'list'
            i += 1
        elif a == '--port' and i + 1 < len(argv):
            try:
                opts['port'] = int(argv[i+1])
            except ValueError:
                print('エラー: --port の値が不正です')
                sys.exit(1)
            i += 2
        elif a == '--protocol' and i + 1 < len(argv):
            p = argv[i+1].upper()
            if p in ('TCP','UDP'):
                opts['protocol'] = p
            else:
                print('エラー: --protocol は TCP か UDP を指定してください')
                sys.exit(1)
            i += 2
        elif a == '--name' and i + 1 < len(argv):
            opts['name'] = argv[i+1]
            i += 2
        else:
            print(f'不明なオプションまたは値が足りません: {a}')
            sys.exit(1)
    if opts['action'] is None:
        opts['action'] = 'add'
    return opts

def main():
    if not is_windows():
        print('このインストーラは Windows 専用です。')
        return
    if not is_admin():
        print('管理者権限が必要です。UAC による昇格を試みます...')
        elevate_to_admin()
    opts = parse_args_manual(sys.argv)
    if opts['action'] == 'add':
        apply_extra_powershell_settings()
        add_rule(port=opts['port'], protocol=opts['protocol'], name=opts['name'])
    elif opts['action'] == 'remove':
        remove_rule(name_pattern=opts['name'] or DEFAULT_NAME_BASE)
    elif opts['action'] == 'list':
        list_rules(name_pattern=opts['name'] or DEFAULT_NAME_BASE)

if __name__ == '__main__':
    main()
'''


def main():
    if not is_windows():
        print("このインストーラは Windows 専用です。")
        return
    root = tk.Tk()
    app = InstallerGUI(root)
    root.mainloop()

if __name__ == "__main__":
    main()
